#!/bin/bash

# 변수 정의
VM_NAME="test-live-migration-vm"
FLAVOR="amphora"  # 필요한 경우 변경
IMAGE="octavia-amphora-haproxy-2024.1.img"  # 필요한 경우 변경
NETWORK="lb-mgmt-net"  # 사용할 네트워크 이름
LOG_FILE="live_migration_result.log"
COMPUTE_NODES=(ds-con1 ds-com1 ds-com2)
VOLUME_NAME="${VM_NAME}-boot-volume"
VOLUME_SIZE=50  # GB

source /root/contrabass-openrc

# 초기화
echo "========== Live Migration Test Started ==========" | tee "$LOG_FILE"
echo "[0] Deleting existing VM and volume..." | tee -a "$LOG_FILE"
openstack server delete "$VM_NAME" --wait 2>/dev/null
openstack volume delete "$VOLUME_NAME" 2>/dev/null
sleep 5

# 1. 볼륨 생성
echo "[1] Creating bootable volume '$VOLUME_NAME' from image '$IMAGE'..." | tee -a "$LOG_FILE"
openstack volume create --image "$IMAGE" --size "$VOLUME_SIZE" --bootable "$VOLUME_NAME" && sleep 45
if [ $? -ne 0 ]; then
    echo "[ERROR] Failed to create volume. Exiting." | tee -a "$LOG_FILE"
    exit 1
fi

# 2. VM 생성 (com1에 고정, 볼륨 부팅 방식)
echo "[2] Creating VM '$VM_NAME' on ${COMPUTE_NODES[0]} from volume '$VOLUME_NAME'..." | tee -a "$LOG_FILE"
openstack --os-compute-api-version 2.74 server create \
    --flavor "$FLAVOR" \
    --network "$NETWORK" \
    --volume "$VOLUME_NAME" \
    --host ${COMPUTE_NODES[0]} \
    "$VM_NAME" --wait

# 결과 확인
if [ $? -eq 0 ]; then
    echo "[OK] VM '$VM_NAME' created successfully." | tee -a "$LOG_FILE"
else
    echo "[ERROR] VM creation failed." | tee -a "$LOG_FILE"
fi

# 2. 마이그레이션 루프 실행
NUM_NODES=${#COMPUTE_NODES[@]}
for (( i=0; i<$NUM_NODES; i++ ))
do
    SRC_NODE=${COMPUTE_NODES[$i]}
    DEST_NODE=${COMPUTE_NODES[$(( (i + 1) % NUM_NODES ))]}

    echo "[${i+2}] Live migrate: $SRC_NODE ➡ $DEST_NODE" | tee -a "$LOG_FILE"
    openstack server migrate --os-compute-api-version 2.74 --live-migration --host "$DEST_NODE" --wait "$VM_NAME"

    if [ $? -ne 0 ]; then
        echo "❌ Migration 실패: $SRC_NODE → $DEST_NODE" | tee -a "$LOG_FILE"
        exit 1
    else
        echo "✅ Migration 성공: $SRC_NODE → $DEST_NODE" | tee -a "$LOG_FILE"
    fi

    # 마이그레이션 완료 후 VM 상태가 ACTIVE이고, DEST_NODE로 실제 이동했는지 확인
    for _ in {1..10}; do
        STATUS=$(openstack server show "$VM_NAME" -f value -c status)
        CURRENT_HOST=$(openstack server show "$VM_NAME" -f value -c OS-EXT-SRV-ATTR:host)
        if [[ "$STATUS" == "ACTIVE" && "$CURRENT_HOST" == "$DEST_NODE" ]]; then
            break
        fi
        sleep 3
    done

    if [[ "$STATUS" != "ACTIVE" ]]; then
        echo "❌ Migration 후 VM 상태 비정상 ($STATUS)" | tee -a "$LOG_FILE"
        exit 1
    fi

    if [[ "$CURRENT_HOST" != "$DEST_NODE" ]]; then
        echo "❌ Migration 후 호스트 변경 실패: 현재=$CURRENT_HOST, 예상=$DEST_NODE" | tee -a "$LOG_FILE"
        exit 1
    fi

    echo "➡️ VM 상태: $STATUS, 현재 호스트: $CURRENT_HOST (정상)" | tee -a "$LOG_FILE"
done

echo "========== Live Migration Test Completed ✅ ==========" | tee -a "$LOG_FILE"
