#!/bin/bash

# 변수 설정
VM_NAME="test-live-migration-vm"
NEW_FLAVOR="8C16G"   # 새로 생성할 flavor 이름
VCPUS=8
RAM=16384  # MB
DISK=0     # Root disk는 volume에서 제공되므로 0GB
LOG_FILE="resize_vm_result.log"

source /root/contrabass-openrc

echo "========== VM Resize Test Started ==========" | tee "$LOG_FILE"

# 0. Flavor 생성
echo "[0] Creating new flavor '$NEW_FLAVOR'..." | tee -a "$LOG_FILE"
openstack flavor show "$NEW_FLAVOR" >/dev/null 2>&1
if [ $? -ne 0 ]; then
    openstack flavor create "$NEW_FLAVOR" --vcpus $VCPUS --ram $RAM --disk $DISK
    if [ $? -ne 0 ]; then
        echo "[ERROR] Failed to create flavor $NEW_FLAVOR" | tee -a "$LOG_FILE"
        exit 1
    fi
else
    echo "  -> Flavor '$NEW_FLAVOR' already exists." | tee -a "$LOG_FILE"
fi

# 1. 현재 Flavor 확인
OLD_FLAVOR=$(openstack server show "$VM_NAME" -f json | jq -r '.flavor.original_name')
echo "[1] Current flavor of $VM_NAME: $OLD_FLAVOR" | tee -a "$LOG_FILE"

# 2. 리사이즈 요청
echo "[2] Resizing VM '$VM_NAME' to flavor '$NEW_FLAVOR'..." | tee -a "$LOG_FILE"
openstack server resize --flavor "$NEW_FLAVOR" "$VM_NAME"
if [ $? -ne 0 ]; then
    echo "[ERROR] Resize command failed." | tee -a "$LOG_FILE"
    exit 1
fi

# 3. VERIFY_RESIZE 상태 대기
echo "[3] Waiting for VM to enter VERIFY_RESIZE state..." | tee -a "$LOG_FILE"
for i in {1..30}; do
    STATUS=$(openstack server show "$VM_NAME" -f value -c status)
    echo "  -> Status: $STATUS" | tee -a "$LOG_FILE"
    if [ "$STATUS" == "VERIFY_RESIZE" ]; then
        break
    fi
    sleep 3
done

if [ "$STATUS" != "VERIFY_RESIZE" ]; then
    echo "[ERROR] VM did not enter VERIFY_RESIZE state. Current: $STATUS" | tee -a "$LOG_FILE"
    exit 1
fi

# 4. 리사이즈 확인
echo "[4] Confirming resize..." | tee -a "$LOG_FILE"
openstack server resize confirm "$VM_NAME"

# 5. ACTIVE 상태 대기
echo "[5] Waiting for VM to return to ACTIVE state..." | tee -a "$LOG_FILE"
for i in {1..30}; do
    STATUS=$(openstack server show "$VM_NAME" -f value -c status)
    echo "  -> Status: $STATUS" | tee -a "$LOG_FILE"
    if [ "$STATUS" == "ACTIVE" ]; then
        break
    fi
    sleep 3
done

if [ "$STATUS" != "ACTIVE" ]; then
    echo "[ERROR] VM did not return to ACTIVE state. Current: $STATUS" | tee -a "$LOG_FILE"
    exit 1
fi

# 6. 새로운 Flavor 확인
NEW_CURRENT_FLAVOR=$(openstack server show "$VM_NAME" -f json | jq -r '.flavor.original_name')
echo "[6] Flavor after resize: $NEW_CURRENT_FLAVOR" | tee -a "$LOG_FILE"

if [ "$NEW_CURRENT_FLAVOR" == "$NEW_FLAVOR" ]; then
    echo "[OK] Resize successful!" | tee -a "$LOG_FILE"
else
    echo "[ERROR] Resize failed. Flavor mismatch." | tee -a "$LOG_FILE"
    exit 1
fi
